#include "pcr_device.h"
#include <iostream>
#include <exception>
#include <cmath>

pcr_device::pcr_device() :
    m_usb(vendorID, vendorName, productID, productName)
{
    try
    {
        set_led(true);
    }
    catch(const exception& ex)
    {
        throw pcr_device_exception("handshake problem", ex);
    }

    cout << "Device info : PCR device ready." << endl;
}

pcr_device::~pcr_device()
{
    try
    {
        set_block_heater(0.0);
    }
    catch(const exception& ex)
    {
        cerr << pcr_device_exception("termination problem (while turning block heater off)", ex).what() << endl;
    }
    try
    {
        set_lid_heater(0.0);
    }
    catch(const exception& ex)
    {
        cerr << pcr_device_exception("termination problem (while turning lid heater off)", ex).what() << endl;
    }
    try
    {
        set_led(false);
    }
    catch(const exception& ex)
    {
        cerr << pcr_device_exception("termination problem (while turning LED off)", ex).what() << endl;
    }

    cout << "Device info : device closed." << endl;
}

uint16_t pcr_device::get_adc(const uint8_t which)
{
    vector<unsigned char>buffer(2);
    try
    {
        int res= m_usb.request(CMD_READ_AD, which, 0, buffer);
        if((size_t)res != buffer.size())
            throw pcr_device_exception("ADC response size invalid.");
    }
    catch(const exception& ex)
    {
        throw pcr_device_exception("ADC reading error.", ex);
    }

    return buffer[0] + buffer[1]*256;
}

double pcr_device::get_block_Tsensor(const unsigned int n_mean) //high level, in degrees
{
    double a(0.0);
    for(unsigned int i(0) ; i < n_mean ; ++i)
        a += get_adc(ADC_BLOCK);
    a /= ((double)adc_max * (double)n_mean);
    return a;
}

double pcr_device::get_lid_Tsensor(const unsigned int n_mean) //high level, in degrees, uncalibrated
{
    double a(0.0);
    for(unsigned int i(0) ; i < n_mean ; ++i)
        a += get_adc(ADC_LID);
    a /= ((double)adc_max * (double)n_mean);

    return a;
}

void pcr_device::set_led(const bool on)
{
    try
    {
        vector<unsigned char> resbuf(2, 0);
        int res= m_usb.request(CMD_LED, (on ? 0xFFFF : 0x0000), 0, resbuf);
        if(res != 1 || resbuf[0] != (on ? 1 : 0))
            throw pcr_device_exception("invalid response");
    }
    catch(const exception& ex)
    {
        throw pcr_device_exception("LED command problem", ex);
    }
}

void pcr_device::set_block_heater(double power)
{
    if(power < -1.0) power= -1.0;
    else if(power > 1.0) power= 1.0;

    uint16_t res_mode;
    uint16_t res_power= round(fabs(power)*(double)pwm_max);
    if(res_power == 0)
        res_mode= HEATER_OFF;
    else if(power > 0)
        res_mode= HEATER_HEAT;
    else
        res_mode= HEATER_COOL;

    vector<unsigned char>buffer(2, res_mode+1);
    try
    {
        int res= m_usb.request(CMD_HEATER_BLOCK, res_power, res_mode, buffer);
        if(res != 1 || buffer[0] != res_mode)
            throw pcr_device_exception("invalid device response.");
    }
    catch(const exception& ex)
    {
        throw pcr_device_exception("block heater command error.", ex);
    }
}

void pcr_device::set_lid_heater(double power)
{
    if(power < -1.0) power= -1.0;
    else if(power > 1.0) power= 1.0;

    uint16_t res_mode;
    uint16_t res_power= round(fabs(power)*(double)pwm_max);
    if(res_power == 0)
        res_mode= HEATER_OFF;
    else if(power > 0)
        res_mode= HEATER_HEAT;
    else
        res_mode= HEATER_OFF;

    vector<unsigned char>buffer(2, res_mode+1);
    try
    {
        int res= m_usb.request(CMD_HEATER_LID, res_power, res_mode, buffer);
        if(res != 1 || buffer[0] != res_mode)
            throw pcr_device_exception("invalid device response.");
    }
    catch(const exception& ex)
    {
        throw pcr_device_exception("lid heater command error.", ex);
    }
}
