#include <iostream>
#include <string>
#include <vector>
#include <deque>
#include <cstdint>
#include <exception>
#include <chrono>
#include <thread>
#include <fstream>
#include <cmath>
#include <algorithm>
#include "pcr_device.h"

using namespace std;
/*
const double thermistor_R1= 1000.0;
const double thermistor_R0= 10000.0;
const double thermistor_T0= 25.0;
const double thermistor_B= 4050.0;
1.0/( log(thermistor_R1 / (thermistor_R0*(1.0/ Tint -1.0)))/thermistor_B + 1.0/(273.15+thermistor_T0) ) - 273.15;
*/

uint64_t ticks()
{
    return std::chrono::system_clock::now().time_since_epoch() / std::chrono::milliseconds(1);
}

void get_PIDparams(const double Tobs, const double target, double &KC, double &KP, double &KI, double &KD, bool up)
{
    KP=KI=KD=0;


    {
        vector<double>KC_poly;
        if(target >= 26.0)
            KC_poly= {1.67332, -0.207962, 0.00957541, -0.000210539, 2.45993e-06, -1.4616e-08, 3.47654e-11};
        else
            KC_poly= {-0.942434, 0.0191819, 3.47245e-05, 2.37447e-05};
        double x= 1.0;
        KC= 0.0;
        for(const double c : KC_poly)
        {
            KC += c*x;
            x *= target;
        }
    }

    if(up)
    {
        KP= 0.075, KI= 0.001, KD= 0.05;
    }
    else //down
    {
        KP= 0.075, KI= 0.001, KD= 0.05;
    }

}

double analog_to_T(const double analog)
{
    double res= 0, x= 1;
    vector<double>polynomial= {56.7129, 2675.63, -27501.9, 133914, -373730, 627364, -626751, 343499, -79547.2};
    for(const double c : polynomial)
    {
        res += c*x;
        x *= analog;
    }
    return res;
}


double stabilize(pcr_device &pcr, const double variation)
{
    cout << "Waiting for stabilization... " << endl;
    double temp;
    deque<double>hist;


    const int waitms= 1000;
    const size_t nhist= 30;

    for(;;)
    {
        temp= pcr.get_block_Tsensor(10);
        hist.push_back(temp);
        if(hist.size() > nhist)
        {
            hist.pop_front();

            double vmin= temp, vmax= temp;
            for(const auto elt : hist)
            {
                if(elt < vmin)
                    vmin= elt;
                if(elt > vmax)
                    vmax= elt;
            }
            if(vmax-vmin <= variation)
            {
                temp= 0.0;
                for(const auto elt : hist)
                    temp += elt;
                temp /= (double)hist.size();
                cout << "Stabilized at : " << temp << " +- " << variation << endl;
                return temp;
            }
            else
                cout << "Stabilization : " << 100.0*variation/(vmax-vmin) << "%" << endl;
        }
        else
            cout << "Stabilization (preload) : " << 100.0*(double)hist.size()/(double)nhist << "%" << endl;

        this_thread::sleep_for(std::chrono::milliseconds(waitms));
    }
}
double read_cleanT(pcr_device &pcr)
{
    cout << "Reading... " << flush;
    const size_t nbuf= 40;
    vector<double>Ts;
    for(size_t i= 0 ; i < nbuf ; ++i)
    {
        double temp= pcr.get_block_Tsensor(12);
        Ts.push_back(temp);
    }
    sort(Ts.begin(), Ts.end());
    double res;
    if(Ts.size()%2 == 0)
        res= (Ts[Ts.size()/2] + Ts[Ts.size()/2-1])/2.0;
    else
        res= Ts[Ts.size()/2];
    cout << "done : " << res << endl;
    return res;
}
void calib_power(pcr_device &pcr, double power, double &analog, double &temperature)
{
    const double variation= 0.0003;

    double mean;
    string tmps;
    while(1)
    {
        cout << "Calibration at power=" << power << endl;
        pcr.set_block_heater(power);
        mean= stabilize(pcr, variation);
        cout << "What temperature are you reading ?" << endl;
        cin >> temperature;
        analog= read_cleanT(pcr);
        if(fabs(analog-mean) > variation)
        {
            cout << "Analog reading is not stable, retrying point..." << endl;
            continue;
        }
        cout << "Data point : Power=" << power << "\tAnalog=" << analog << "\tTemperature=" << temperature << endl;
        break;
    }
}
void calibrate(pcr_device &pcr)
{
    struct dta
    {
        double power, analog, temperature;
    };

    ofstream outf("out.txt");
    outf << "Power\tAnalog\tTemperature" << endl;

    vector< dta >data;
    dta resdta;

    cout << "Calibrating block..." << endl;

    cout << "Running first pass..." << endl;
    for(int i= 16 ; i <= 20 ; ++i)
    {
        resdta.power= -1.0 + 0.1*(double)i;
        calib_power(pcr, resdta.power, resdta.analog, resdta.temperature);
        data.push_back(resdta);
        outf << resdta.power << "\t" << resdta.analog << "\t" << resdta.temperature << endl;
    }
    cout << "First pass finished." << endl;

    cout << "Running detailed pass..." << endl;
    const double maxdelta_analog= 0.0001, mindelta_power= 0.00015, maxdelta_T=0.11;
    double maxdelta;
    size_t resindex;
    while(1)
    {
        maxdelta= 0.000;
        resindex= 0;

        for(size_t i= 1 ; i < data.size() ; ++i)
        {
            if(data[i].temperature>=50 && data[i].temperature <= 100 && (fabs(data[i].temperature - data[i-1].temperature) > maxdelta_T) && (data[i].power - data[i-1].power >= mindelta_power))
            {
                if(fabs(data[i].temperature - data[i-1].temperature) > maxdelta)
                    maxdelta= fabs(data[i].temperature - data[i-1].temperature),
                    resindex= i;
            }
        }
        if(resindex > 0)
        {
            cout << "deltaT=" << maxdelta << endl;
            resdta.power= (data[resindex].power + data[resindex-1].power)/2.0;
            calib_power(pcr, resdta.power, resdta.analog, resdta.temperature);
            data.insert(data.begin()+resindex, dta(resdta));
            outf << resdta.power << "\t" << resdta.analog << "\t" << resdta.temperature << endl;
        }
        else
            break;
    }
    cout << "Detailed pass finished. Calibration done." << endl;
}


int main()
{
    vector<double>targets;
    vector<double>times;

    targets.push_back(95), times.push_back(120);
    for(int i= 0 ; i < 30 ; ++i)
    {
        targets.push_back(95), times.push_back(30);
        targets.push_back(60), times.push_back(30);
        targets.push_back(72), times.push_back(180);
    }
    targets.push_back(72), times.push_back(600);
    targets.push_back(12), times.push_back(600);

    cout << "Time\tObs\tTarget\tCmd" << endl;
    try
    {
        pcr_device pcr;

        calibrate(pcr);

        try
        {
            double target;
            double res, Tobs, err, prev_err= 0, integral=0, derivative, Tinitial;
            double t, dt;
            unsigned int targetid= 0;

            double resp, resi, resd;

            double KC, KP, KI, KD;

            uint64_t curticks, ticks_start, prev_ticks, ticks_init;
            curticks= ticks_start= prev_ticks= ticks_init= ticks();

            double deltatemp= 1.0;
            double Tint;

            bool first= true, up;
            bool stabilizing= true;
            for(int i=0 ; ; ++i)
            {
                curticks= ticks();
                dt= (curticks-prev_ticks)/1000.0;
                prev_ticks= curticks;

                target= targets[targetid];
                Tint= pcr.get_block_Tsensor(10);
                Tobs= analog_to_T(Tint);
                if(i == 0)
                    Tinitial= Tobs;
                err= target-Tobs;

                if(targetid == 0) up= (target > Tinitial);
                else up= (target > targets[targetid-1]);

                if(stabilizing)
                {
                    if((up && Tobs >= target-deltatemp) || (!up && Tobs <= target+deltatemp))
                    {
                        stabilizing= false;
                        ticks_start= curticks= ticks();
                        integral= 0.0;
                    }
                    else
                        t= 0;
                }
                else
                    t= (curticks-ticks_start)/1000.0;

                if(t >= times[targetid])
                {
                    targetid++;
                    integral= 0;
                    stabilizing= true;
                    if(targetid >= targets.size())
                        break;
                }

                //Derivative
                if(first)
                    derivative= 0.0;
                else
                    derivative= (err - prev_err)/dt;
                prev_err= err;

                //integral
                integral += (err*dt)/(0.01+err*err);

                //result
                get_PIDparams(Tobs, target, KC, KP, KI, KD, up);
                resp= KP*err, resi= KI*integral, resd= KD*derivative;

                if(resp > 2.0) err= 2.0/KP; else if(resp < -2.0) err= -2.0/KP;
                if(resi > 1.0) integral= 1.0/KI; else if (resi < -1.0) integral= -1.0/KI;
                if(resd > 1.0) derivative= 1.0/KD; else if (resd < -1.0) derivative= -1.0/KD;

                resp= KP*err, resi= KI*integral, resd= KD*derivative;
                res= KC + resp + resi + resd;

                pcr.set_block_heater(res);
                cout << (ticks() - ticks_init)/1000.0 << "\t" << Tobs << "\t" << target << "\t" << res << "\tc=" << KC << "\tp=" << resp << "\ti=" << resi << "\td=" << resd << "\t" << Tint<< endl;

                this_thread::sleep_for(std::chrono::milliseconds(20));
                first= false;
            }

        }
        catch(const exception& ex)
        {
            cerr << ex.what() << endl;
            return EXIT_FAILURE;
        }
    }
    catch(const exception& ex)
    {
        cerr << ex.what() << endl;
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
