#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>

#include "usbdrv.h"
#include <util/delay.h>

#define BIT(bitno) (1<<bitno)
#define SET(port,pin) port |= BIT(pin)
#define RST(port,pin) port &= ~BIT(pin)

#define CMD_LED 0
#define CMD_READ_AD 1
#define CMD_HEATER_BLOCK 2
#define CMD_HEATER_LID 3

#define PIN_LED PD0
#define PIN_LID PD5
#define PIN_PELTIER1 PD6
#define PIN_PELTIER2 PD7

#define HEATER_OFF  0
#define HEATER_HEAT 1
#define HEATER_COOL 2

#define PWM_MAXVALUE 8191

#define ADC_BLOCK 0
#define ADC_LID   1

uint8_t block_mode= HEATER_OFF, lid_mode= HEATER_OFF;
uint16_t block_intensity= 0, lid_intensity= 0;

typedef union
{
    uint16_t word;
    uint8_t bytes[2];
} adc_value_t;

adc_value_t adc_value;


inline void timer_event()
{
    static uint16_t counter= 0;
    while(++counter > PWM_MAXVALUE)
    	counter= 0;
    switch(block_mode)
    {
    case HEATER_HEAT :
        RST(PORTD, PIN_PELTIER2);
        if(counter <= block_intensity)
            SET(PORTD, PIN_PELTIER1);
        else
            RST(PORTD, PIN_PELTIER1);
        break;
    case HEATER_COOL :
        RST(PORTD, PIN_PELTIER1);
        if(counter <= block_intensity)
            SET(PORTD, PIN_PELTIER2);
        else
            RST(PORTD, PIN_PELTIER2);
        break;
    default :
            RST(PORTD, PIN_PELTIER1);
            RST(PORTD, PIN_PELTIER2);
        break;
    }
    switch(lid_mode)
    {
    case HEATER_HEAT :
        if(counter <= lid_intensity)
            SET(PORTD, PIN_LID);
        else
            RST(PORTD, PIN_LID);
        break;
    default :
            RST(PORTD, PIN_LID);
        break;
    }
}

void doevents()
{
    wdt_reset();

    if(TCNT1 >= 2) //2000000Hz/(2*8191)=122Hz PWM
    {
        timer_event();
        TCNT1= 0;
    }
}

inline void get_adc(const uint8_t which)
{
    uint8_t i;
    adc_value_t tmpadc;
    switch(which)
    {
    case 0 :
        RST(ADMUX, MUX0);
        break;
    case 1:
        SET(ADMUX, MUX0);
        break;
    default :
        adc_value.word= 0;
        return;
    }
    adc_value.word= 0;
    for(i= 0 ; i < 64 ; ++i) //max value : 64*1023=65472, total duration~0.6ms
    {
        SET(ADCSRA, ADSC); //run ADC
        while(ADCSRA & BIT(ADSC))
            doevents(); //wait for conversion to finish
        tmpadc.bytes[0]= ADCL;
        tmpadc.bytes[1]= ADCH;
        adc_value.word += tmpadc.word;
    }
}

inline void init_adc()
{
    ADCSRA |= BIT(ADPS2) | BIT(ADPS1) | BIT(ADPS0); // Set ADC prescalar to 128 - 125KHz sample rate @ 16MHz
    SET(ADMUX, REFS0); // Set ADC reference to AVCC
    // No MUX values need to be changed to use ADC0 : ADC0 activated by default
    SET(ADCSRA, ADEN); // Enable ADC
    //SET(ADCSRA, ADSC); //run ADC
}

USB_PUBLIC uchar usbFunctionSetup(uchar data[8])
{
    uchar res_size= 0;
    usbRequest_t *rq = (void *)data;
    static uchar replyBuf[2];

    usbMsgPtr= replyBuf;

    switch(rq->bRequest)
    {
    case CMD_LED : //LED
        if(rq->wValue.bytes[0])
            SET(PORTD, PIN_LED);
        else
            RST(PORTD, PIN_LED);
        replyBuf[0]= (PORTD & BIT(PIN_LED) ? 1 : 0);
        res_size= 1;
        break;
    case CMD_READ_AD :
        get_adc(rq->wValue.bytes[0]);
        replyBuf[0]= adc_value.bytes[0];
        replyBuf[1]= adc_value.bytes[1];
        res_size= 2;
        break;
    case CMD_HEATER_BLOCK : //peltier : wIndex[0] for mode ; wValue for pwm
        block_mode= rq->wIndex.bytes[0];
        block_intensity= rq->wValue.word;
        replyBuf[0]= block_mode;
        res_size= 1;
        break;
    case CMD_HEATER_LID : //lid heater : wIndex[0] for mode ; wValue for pwm
        lid_mode= rq->wIndex.bytes[0];
        lid_intensity= rq->wValue.word;
        replyBuf[0]= lid_mode;
        res_size= 1;
        break;
    }
    return res_size;
}

inline void init_usb()
{
    PORTD= 0;          // no pullups on USB pins
    //0 : in ; 1 : out
    DDRD= 0b00000000 & ~USBMASK; //usb as input, others as output

    usbDeviceDisconnect();  // enforce re-enumeration, do this while interrupts are disabled!
    {
        uint8_t i;
        for(i= 0 ; i < 250 ; ++i)
        {
            wdt_reset();
            _delay_ms(2);
        }
    }
    usbDeviceConnect();
    usbInit();
    sei();
}


int main(void)
{
    wdt_enable(WDTO_1S);

    init_usb();

    //Setup ports
    DDRD |= BIT(PIN_LED)       //output : LED
         |  BIT(PIN_LID)       //output : lid
         |  BIT(PIN_PELTIER1)  //output : peltier 1
         |  BIT(PIN_PELTIER2); //output : peltier 2
    PORTD &= ( ~BIT(PIN_PELTIER1) & ~BIT(PIN_PELTIER2) ); //Peltier off
    PORTD &= ~BIT(PIN_LID); //Lid heater off
    PORTD &= ~BIT(PIN_LED); //LED off

    init_adc();

    //init timer : 16MHz/8 = 2MHz
    TCCR1B |= (1 << CS11);

    //main loop
    for(;;)
    {
        doevents();
        usbPoll();
    }

    return 0;
}
