#include "usb_wrapper.h"
#include <iostream>

usb_wrapper::usb_wrapper(const int vendor, const string &vendorName, const int product, const string &productName) :
    m_handle(NULL)
{
    int res;

    res= libusb_init(NULL);
    if(res < 0) //error
    {
        throw usbexception("cannot initialize interface.");
        return;
    }

    libusb_set_debug(NULL, 3); // set debug level

    //find device
    libusb_device **devs;
    ssize_t ndev= libusb_get_device_list(NULL, &devs);
    if(ndev < 0)
    {
        throw usbexception("cannot get USB device list.");
        return;
    }

    libusb_device *dev;
    libusb_device_descriptor desc;
    libusb_device_handle *tmp_handle;
    unsigned char buffer[256];
    for(ssize_t i(0) ; i < ndev ; ++i)
    {
        dev= devs[i];
        //get descriptor
        res= libusb_get_device_descriptor(dev, &desc);
        if(res)
        {
            cerr << "USB warning : cannot get descriptors for an USB device." << endl;
            continue;
        }
        //check vendor and product numbers
        if(desc.idVendor != vendor || desc.idProduct != product)
            continue;

        //open device
        res= libusb_open(dev, &tmp_handle);
        if(res)
        {
            cerr << "USB warning : cannot open an USB device." << endl;
            continue;
        }

        //get strings
        res= libusb_get_string_descriptor_ascii(tmp_handle, desc.iManufacturer, buffer, sizeof(buffer)-1);
        if(res < 0)
        {
            libusb_close(tmp_handle);
            cerr << "USB warning : cannot get manufacturer string for an USB device." << endl;
            continue;
        }
        buffer[res]= 0;
        if(vendorName != string((char*)buffer))
            continue;

        res= libusb_get_string_descriptor_ascii(tmp_handle, desc.iProduct, buffer, sizeof(buffer)-1);
        if(res < 0)
        {
            libusb_close(tmp_handle);
            cerr << "USB warning : cannot get product string for an USB device." << endl;
            continue;
        }
        buffer[res]= 0;
        if(productName != string((char*)buffer))
            continue;

        m_handle= tmp_handle;
        break;
    }
    libusb_free_device_list(devs, 1);
    if(!m_handle)
    {
        throw usbexception("cannot find USB device.");
        return;
    }

    cout << "USB info : USB link established." << endl;
}

size_t usb_wrapper::request(const uint8_t bRequest, const uint16_t wValue, const uint16_t wIndex, vector<unsigned char>&retbuf, unsigned int timeout)
{
    int res= libusb_control_transfer(m_handle, LIBUSB_REQUEST_TYPE_VENDOR | LIBUSB_RECIPIENT_DEVICE | LIBUSB_ENDPOINT_IN,
                                     bRequest, wValue, wIndex, retbuf.data(), retbuf.size(), timeout);
    if(res >= 0)
        return res; //OK
    switch(res)
    {
    case LIBUSB_ERROR_TIMEOUT :
        throw usbexception("request timeout.");
    case LIBUSB_ERROR_PIPE :
        throw usbexception("control request unsupported by device.");
    case LIBUSB_ERROR_NO_DEVICE :
        throw usbexception("device disconnected.");
    default :
        throw usbexception("problem while sending USB request.");
    }
    return 0;
}

usb_wrapper::~usb_wrapper()
{
    if(m_handle)
        libusb_close(m_handle);
    m_handle= NULL;
    cout << "USB info : USB link closed." << endl;
}
