#include <iostream>
#include <string>
#include <vector>
#include <deque>
#include <cstdint>
#include <exception>
#include <chrono>
#include <thread>
#include <fstream>
#include <cmath>
#include <algorithm>
#include "pcr_device.h"

using namespace std;

uint64_t ticks()
{
    return std::chrono::system_clock::now().time_since_epoch() / std::chrono::milliseconds(1);
}

void get_PIDparams(const double obs, const double target, double &KC, double &KP, double &KI, double &KD, bool up)
{
    KC= 0;
    KP= 0.075;
    KI= 0.001;
    KD= 0.05;
}

double A_to_T(const double input)
{
    double res= 0, x= 1;
    vector<double>polynomial= {0.662002,3379.17,-30600,138765,-368807,597239,-580302,311063,-70759.3};
    for(const double c : polynomial)
    {
        res += c*x;
        x *= input;
    }
    return res;
}
double T_to_P(const double input)
{
    double res= 0, x= 1;
    vector<double>polynomial;
    if(input < 25.0)
        polynomial= {-0.879761,0.0296994,-0.000163702,-7.46044e-05,1.31263e-05,-7.04964e-07,1.44921e-08,-6.16827e-11};
    else
        polynomial= {-0.941596,0.0993558,-0.0044541,0.000112071,-1.59508e-06,1.29228e-08,-5.53686e-11,9.7232e-14};
    for(const double c : polynomial)
    {
        res += c*x;
        x *= input;
    }
    return res;
}
double A_to_P(const double input)
{
    double res= 0, x= 1;
    vector<double>polynomial;
    if(input > 0.914069)
        polynomial= {1772.82,-5528.53,5650.62,-1776.86,-119.668};
    else
        polynomial= {1.07533,3.59846,-25.348,-169.494,2131.48,-9501.22,23587.4,-35455,32086.8,-16105.9,3446.94};
    for(const double c : polynomial)
    {
        res += c*x;
        x *= input;
    }
    return res;
}


int main()
{
    vector<double>targets;
    vector<double>times;

    {
        double tmp_target, tmp_time;
        ifstream ifs("seq.txt");
        while(ifs >> tmp_target >> tmp_time)
            targets.push_back(tmp_target),
            times.push_back(tmp_time);
    }

    cout << "Time\tObs\tTarget\tCmd" << endl;
    try
    {
        pcr_device pcr;

        try
        {
            double target;
            double res, obs, err, prev_err= 0, integral=0, derivative, initial;
            double t, dt;
            unsigned int targetid= 0;

            double resp, resi, resd;

            double KC, KP, KI, KD;

            uint64_t curticks, ticks_start, prev_ticks, ticks_init;
            curticks= ticks_start= prev_ticks= ticks_init= ticks();

            double deltatemp= 1.0;
            double Tint, Ttarget;

            bool first= true, up;
            bool stabilizing= true;
            for(int i=0 ; ; ++i)
            {
                curticks= ticks();
                dt= (curticks-prev_ticks)/1000.0;
                prev_ticks= curticks;

                Ttarget= targets[targetid];
                target= T_to_P(targets[targetid]);
                Tint= pcr.get_block_Tsensor(10);
                obs= A_to_P(Tint);
                if(i == 0)
                    initial= obs;
                err= target-obs;

                if(targetid == 0) up= (target > initial);
                else up= (target > T_to_P(targets[targetid-1]));

                if(stabilizing)
                {
                    if((up && obs >= target-deltatemp) || (!up && obs <= target+deltatemp))
                    {
                        stabilizing= false;
                        ticks_start= curticks= ticks();
                        integral= 0.0;
                    }
                    else
                        t= 0;
                }
                else
                    t= (curticks-ticks_start)/1000.0;

                if(t >= times[targetid])
                {
                    targetid++;
                    integral= 0;
                    stabilizing= true;
                    if(targetid >= targets.size())
                        break;
                }

                //Derivative
                if(first)
                    derivative= 0.0;
                else
                    derivative= (err - prev_err)/dt;
                prev_err= err;

                //integral
                integral += (err*dt)/(0.01+err*err);

                //result
                get_PIDparams(obs, target, KC, KP, KI, KD, up);
                resp= KP*err, resi= KI*integral, resd= KD*derivative;

                if(resp > 2.0) err= 2.0/KP; else if(resp < -2.0) err= -2.0/KP;
                if(resi > 1.0) integral= 1.0/KI; else if (resi < -1.0) integral= -1.0/KI;
                if(resd > 1.0) derivative= 1.0/KD; else if (resd < -1.0) derivative= -1.0/KD;

                resp= KP*err, resi= KI*integral, resd= KD*derivative;
                res= KC + resp + resi + resd;

                pcr.set_block_heater(res);
                cout << (ticks() - ticks_init)/1000.0 << "\t" << A_to_T(Tint) << "\t" << Ttarget << "\t" << res << "\tc=" << KC << "\tp=" << resp << "\ti=" << resi << "\td=" << resd << endl;

                this_thread::sleep_for(std::chrono::milliseconds(1000));
                first= false;
            }

        }
        catch(const exception& ex)
        {
            cerr << ex.what() << endl;
            return EXIT_FAILURE;
        }
    }
    catch(const exception& ex)
    {
        cerr << ex.what() << endl;
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
