function [Record]=gillespieV2(Nmax,Tend,k)
%This function implements the Gillespie algorithm for the simulation of
%random events. In the case, cell division and plasmid repartition bewteen
%"daughter cells".

%This section sets the parameters of the probabilistic law representing the
%random events of division and plasmid repartition

%Cell division : An Erlang distribution maps the disparities in
%the time separating two mitosis events.
%The density of Erlang's dis is given by :
%f(x)=[ lambda^k.x^(k-1).e^-(lambda.x)]
% "divided by"     (k-1)!
%Why use this law : The Erlang distribution is the distribution of the sum of k independent identically distributed random variables each having an exponential distribution.
%we want E[tm]=1 to ensure the mean division time is 1. This implies to set
%the lambda parameter to the fixed value : lambda=k, leaving only one parmeter, the "shape" parameter k.
%You can use the function ErlangPlot(k) to visualize the density.
% As Erlang(k,lambda) is equivalant to gamma(k,1/lamdba) we use
% gamrnd(k,1/k) to generate the time interval of a cell at birth.
%k=3;


%Mean time in hours between mitosis : a linear interpolation of the
%following points ([Time;Rate])
PointsDivRate=[0 2 2.68 3.19 5.7 9.8 12.15 14.74;1.3 3 3.6 2.21 1.055 0.6 0.5 0.05]';

MeanCDivRate=@(t)(LinInter(t,PointsDivRate));
CellDivMean0=MeanCDivRate(1);


%Plasmid repartition
%When a division occurs, the amount of plasmid is divided in proportions
%following a gaussian law of mean 0.5 (fifty fifty repartition) and std
%deviation of 0.05 (for example)
Pmean=0.5;
Pstd=@(t)(0.1);
UseUniform=1;%To use an uniform repartition in plasmid splitting
bounds=[0.4,0.9];%bounds for disparity between 0 and 1


%Initial amount of plasmids
P0=3*10^7;%Quick estimation with Thomas


%Ending parameters
%We can choose one of the following :
%Nmax=1000;%Max number of iteration
%Cmax=2000;%Max number of cells

%Metaparmeter
RatesamplingCoef=0.03;
%This rate when >0 makes less mesures after a while to be able to go
%further, But is is less accurate ! If =0 leaves a classic gillespie


%This array will carry over all the information about our cells. Each
%column is a different cell and the rows represent respectively :
%birthTime/Amount of plasmid/Time to next division.
Cells=[0;P0;gamrnd(k,1/k)/CellDivMean0];
History=[];
%Will carry over the cell which have divided (and became indistinguishable)
%for visualising the proccess after. rows mean plasmid amount/Time at
%division

%Vector for statistic information
PopulationSize=[0;1];
MeanPlasmidAmount=[0;P0];
StdPlasmidAmount=[0;0];
%Used for statistics on extreme values
MinMaxRatio=[0;0];
MinMaxSampling=0:1:round(Tend)+1;
MinMaxThres=MinMaxSampling(1);
MinMaxIndex=1;

Time=0;
N=0;



h=waitbar(0,'iterating');
while N<Nmax && Time<Tend
%Main loop
Wn=MeanCDivRate(Time)*exp(-Time*RatesamplingCoef)*size(Cells,2);%Cumulative rate of division

TStep=-log(1-rand(1,1))/Wn;%Random picking of the observation time
if N==0
    TStep=Cells(3,:)+0.0001;
end
Time=Time+TStep;
Cells(3,:)=Cells(3,:)-TStep;
IndexCellsToDivide=Cells(3,:)<0;
if ~any(IndexCellsToDivide)
    continue
    %If no cell has to divide, the loop is skiped
end
IndexCellsToDivide=find(IndexCellsToDivide);
CellsToDivide=Cells(:,IndexCellsToDivide);
NCellDiv=size(CellsToDivide,2);


%Plasmid repartition
if ~UseUniform
    PlasmidRepartitionVector=min(max(normrnd(Pmean,Pstd(Time),1,NCellDiv),zeros(1,NCellDiv)),ones(1,NCellDiv));
else
    PlasmidRepartitionVector=min(max(unifrnd(bounds,1,NCellDiv),zeros(1,NCellDiv)),ones(1,NCellDiv));
end

%The min max trick ensures we don't have wired repartitions
PlasmidsMotherCells=round(CellsToDivide(2,:).*PlasmidRepartitionVector);%Not a dot product
PlasmidsDaughterCells=CellsToDivide(2,:)-PlasmidsMotherCells;



%Next division time
NextDivTimes=gamrnd(k,1/k,2,NCellDiv)*(1/MeanCDivRate(Time));



%Update the population
Cells(:,IndexCellsToDivide)=[Time*ones(1,NCellDiv);PlasmidsMotherCells;NextDivTimes(1,:)];
Cells(:,(end+1):(end+NCellDiv))=[Time*ones(1,NCellDiv);PlasmidsDaughterCells;NextDivTimes(2,:)];


%Recordings : this slows the process and consume memory, watch it !
History=[History [CellsToDivide(2,:);Time*ones(1,NCellDiv)]];
PopulationSize(:,N+1)=[Time;size(Cells,2)];
MeanPlasmidAmount(:,N+1)=[Time;mean(Cells(2,:))];
StdPlasmidAmount(:,N+1)=[Time;std(Cells(2,:))];
if Time>MinMaxThres
    MinMaxThres=MinMaxSampling(MinMaxIndex+1);
    MinMaxIndex=MinMaxIndex+1;
    MinMaxRatio(:,MinMaxIndex)=[Time;min(Cells(2,:))/(max(Cells(2,:))+min(Cells(2,:)))];
end

%End of a step
N=N+1;
waitbar(N/Nmax,h);
end
close(h);

Record={History,PopulationSize,MeanPlasmidAmount,StdPlasmidAmount,MinMaxRatio};

%Cell growth reference. If simulations need to be comparable with time data,
%make sure the growth fits this, which, because of the stochasticity can
%require several tries.
refpoints=[0 1.56 1.8 2 2.68 3.19 5.79 9.8 12.15 14.74;1 2 4 8 16 32 2500 40000 100000 180000];


figure
plot(refpoints(1,:),refpoints(2,:),'g');hold on;
plot(PopulationSize(1,:),PopulationSize(2,:),'r');hold off;
title({'Population size over time',num2str(PointsDivRate(2,:))});
xlabel('Time (h)');
ylabel('# of plasmids');
figure
plot(MeanPlasmidAmount(1,:),log10(MeanPlasmidAmount(2,:)),'k');
hold on;
plot(StdPlasmidAmount(1,:),log10(StdPlasmidAmount(2,:)),'b');
title('blue : std, black : mean');
xlabel('Time (h)');
ylabel('# of cells, log base 10');
hold off;



%end